PARAM( 
	[string]$Principal = $(throw "`nMissing -Principal DOMAIN\Group"), 
	$Computers = $(throw "`nMissing -Computers ('server01','server02')"))

# USAGE: 
# .\Set-RemotePermission-DCOM.ps1 -Principal "DOMAIN\<group or username>" -Computers ('<server1>', '<server2>',...)
#
# EXAMPLE:
# .\Set-RemotePermission-DCOM.ps1 -Principal "DOMAIN\LG-Citrix-Admins" -Computers ('CTX_DC001', 'CTX_DC002')
#
# Inspired by Karl Mitschke's post:
# http://unlockpowershell.wordpress.com/2009/11/20/script-remote-dcom-wmi-access-for-a-domain-user/
#
# And inspired Brad Turner's post:
# http://social.technet.microsoft.com/Forums/en-US/ilm2/thread/5db2707c-87c9-4bb2-a0eb-912363e2814a/


function get-sid
{
 PARAM ($DSIdentity)
 $ID = new-object System.Security.Principal.NTAccount($DSIdentity)
 return $ID.Translate( [System.Security.Principal.SecurityIdentifier] ).toString()
}

$sid = get-sid $Principal

#DefaultLaunchPermission - Local Launch, Remote Launch, Local Activation, Remote Activation
$DCOMSDDLDefaultLaunchPermission = "A;;CCDCLCSWRP;;;$sid"

#DefaultAccessPermision - Local Access, Remote Access
$DCOMSDDLDefaultAccessPermision = "A;;CCDCLC;;;$sid"

#PartialMatch
$DCOMSDDLPartialMatch = "A;;\w+;;;$sid"

foreach ($strcomputer in $computers)
{
 write-host "`nWorking on $strcomputer with principal $Principal ($sid):"
 # Get the respective binary values of the DCOM registry entries
 $Reg = [WMIClass]"\\$strcomputer\root\default:StdRegProv"
 $DCOMDefaultLaunchPermission = $Reg.GetBinaryValue(2147483650,"software\microsoft\ole","DefaultLaunchPermission").uValue
 $DCOMDefaultAccessPermission = $Reg.GetBinaryValue(2147483650,"software\microsoft\ole","DefaultAccessPermission").uValue

 # Convert the current permissions to SDDL
 write-host "`tConverting current permissions to SDDL format..."
 $converter = new-object system.management.ManagementClass Win32_SecurityDescriptorHelper
 $CurrentDCOMSDDLDefaultLaunchPermission = $converter.BinarySDToSDDL($DCOMDefaultLaunchPermission)
 $CurrentDCOMSDDLDefaultAccessPermission = $converter.BinarySDToSDDL($DCOMDefaultAccessPermission)


 # Build the new permissions
 if (($CurrentDCOMSDDLDefaultLaunchPermission.SDDL -match $DCOMSDDLPartialMatch) -and ($CurrentDCOMSDDLDefaultLaunchPermission.SDDL -notmatch $DCOMSDDLDefaultLaunchPermission))
 {
   $NewDCOMSDDLDefaultLaunchPermission = $CurrentDCOMSDDLDefaultLaunchPermission.SDDL -replace $DCOMSDDLPartialMatch, $DCOMSDDLDefaultLaunchPermission
 }
 else
 {
   $NewDCOMSDDLDefaultLaunchPermission = $CurrentDCOMSDDLDefaultLaunchPermission.SDDL + "(" + $DCOMSDDLDefaultLaunchPermission + ")"
 }

 if (($CurrentDCOMSDDLDefaultAccessPermission.SDDL -match $DCOMSDDLPartialMatch) -and ($CurrentDCOMSDDLDefaultAccessPermission.SDDL -notmatch $DCOMSDDLDefaultAccessPermision))
 {
   $NewDCOMSDDLDefaultAccessPermission = $CurrentDCOMSDDLDefaultAccessPermission.SDDL -replace $DCOMSDDLPartialMatch, $DCOMSDDLDefaultAccessPermision
 }
 else
 {
   $NewDCOMSDDLDefaultAccessPermission = $CurrentDCOMSDDLDefaultAccessPermission.SDDL + "(" + $DCOMSDDLDefaultAccessPermision + ")"
 }



 # Convert SDDL back to Binary
 write-host "`tConverting SDDL back into binary form..."
 $DCOMbinarySDDefaultLaunchPermission = $converter.SDDLToBinarySD($NewDCOMSDDLDefaultLaunchPermission)
 $DCOMconvertedPermissionDefaultLaunchPermission = ,$DCOMbinarySDDefaultLaunchPermission.BinarySD

 $DCOMbinarySDDefaultAccessPermission = $converter.SDDLToBinarySD($NewDCOMSDDLDefaultAccessPermission)
 $DCOMconvertedPermissionsDefaultAccessPermission = ,$DCOMbinarySDDefaultAccessPermission.BinarySD

 # Apply the changes
 write-host "`tApplying changes..."
 if ($CurrentDCOMSDDLDefaultLaunchPermission.SDDL -match $DCOMSDDLDefaultLaunchPermission)
 {
   write-host "`t`tCurrent DefaultLaunchPermission matches desired value."
 }
 else
 {
   $result = $Reg.SetBinaryValue(2147483650,"software\microsoft\ole","DefaultLaunchPermission", $DCOMbinarySDDefaultLaunchPermission.binarySD)
   if($result.ReturnValue='0'){write-host "  Applied DefaultLaunchPermission complete."}
 }

 if ($CurrentDCOMSDDLDefaultAccessPermission.SDDL -match $DCOMSDDLDefaultAccessPermision)
 {
   write-host "`t`tCurrent DefaultAccessPermission matches desired value."
 }
 else
 {
   $result = $Reg.SetBinaryValue(2147483650,"software\microsoft\ole","DefaultAccessPermission", $DCOMbinarySDDefaultAccessPermission.binarySD)
   if($result.ReturnValue='0'){write-host "  Applied DefaultAccessPermission complete."}

 }
}
#----------------------------------------------------------------------------------------------------------
 trap 
 { 
 $exMessage = $_.Exception.Message
 if($exMessage.StartsWith("L:"))
 {write-host "`n" $exMessage.substring(2) "`n" -foregroundcolor white -backgroundcolor darkblue}
 else {write-host "`nError: " $exMessage "`n" -foregroundcolor white -backgroundcolor darkred}
 Exit
 }
#----------------------------------------------------------------------------------------------------------