# This script was inspired by Ingmar Verheij's post on configuring IIS for App-V 5 streaming over HTTP.
# http://www.ingmarverheij.com/microsoft-app-v-5-0-streaming-via-http/
# The script is intended to be used during the server build process, via MDT or some other such method.
# Author: Patrick Seymour

Set-StrictMode -Version "2.0"

# See if the Web Administration module is available.
if ((Get-Module -ListAvailable | Where-Object { $_.Name -eq "WebAdministration" }) -eq $null)
{
    Write-Error -Message "WebAdministration module does not appear to be available. Quitting."
    exit 1
}

# Load the Web Administration module if it is not already loaded.
if ((Get-Module -Name "WebAdministration") -eq $null)
{
    Import-Module -Name "WebAdministration"
}

# If the Web Administration module has not been loaded by this point, just give up. :)
if ((Get-Module -Name "WebAdministration") -eq $null)
{
    Write-Error -Message "Unable to load WebAdministration module. Quitting."
    exit 1
}



# Add a MIME type for .app files.
Add-WebConfigurationProperty //staticContent -Name collection -value @{fileExtension='.appv'; mimeType='application/appv'}



# This will be the name of the folder housing the App-V content.
[string]$ContentFolderName = "App-V 5 Content"

# We are going to query WMI for the local logical disk with the most free space.
# Setup a basic query here.
[string]$wmiFilter = "(DriveType = 3)"

# If there is more than one local disk, we want to filter out the disk that contains the operating system.
if ((@(Get-WmiObject -Class Win32_LogicalDisk -Filter "DriveType = 3")).Count -gt 1)
{
    $wmiFilter = ("((DriveType = 3) AND (Name <> '{0}'))" -f ([System.Environment]::ExpandEnvironmentVariables("%SystemDrive%")))
}

# Get the name (drive letter) of the logical disk with the most free space.
[string]$nonSysDriveLetter = (Get-WmiObject -Class "Win32_LogicalDisk" -Filter $wmiFilter | Sort-Object -Property FreeSpace -Descending | Select-Object -First 1).Name

# Build the full path of the App-V content folder.
[string]$contentFolderPath = Join-Path $nonSysDriveLetter $ContentFolderName

# Create the App-V content folder, if it does not already exist.
if (!(Test-Path $contentFolderPath)) { New-Item -Path $contentFolderPath -ItemType "Directory" | Out-Null }



# In the default web site, create a virtual directory for the App-V content.
Get-Website -Name "Default Web Site" | Select-Object -ExpandProperty Name | % { New-WebVirtualDirectory -Site $_ -Name "AppVContent" -PhysicalPath $contentFolderPath }



# Set the maximum size of a file that can be cached. (See http://support.microsoft.com/kb/954864)
Set-ItemProperty -Path "Registry::HKLM\System\CurrentControlSet\Services\InetInfo\Parameters" -Name "MaxCachedFileSizeInMB" -Value 4096 -Type DWord

# Set the maximum length of time (in seconds) that objects are held in cached memory. (See http://support.microsoft.com/kb/954864)
Set-ItemProperty -Path "Registry::HKLM\System\CurrentControlSet\Services\InetInfo\Parameters" -Name "ObjectCacheTTL" -Value 600 -Type DWord



# Set the number of requests necessary to determine that a URL is frequently hit.  (See http://msdn.microsoft.com/en-us/library/ms692441.aspx)
Set-WebConfigurationProperty -Filter "/system.webServer/serverRuntime" -Name "frequentHitThreshold" -Value 1

# Set the amount of time necessary to determine that a URL is frequently hit. (See http://msdn.microsoft.com/en-us/library/ms692441.aspx)
Set-WebConfigurationProperty -Filter "/system.webServer/serverRuntime" -Name "frequentHitTimePeriod" -Value 00:00:01



# Return a success error code.
# Note: This is not strictly necessary. It is useful if another script is calling this one.
exit 0